/*-
******************************************************************************
******************************************************************************
**
**  ARCHIVE HEADER INFORMATION
**
**  @C-file{
**      FILENAME    = "unix.c",
**      VERSION     = "1.00",
**      DATE        = "",
**      TIME        = "",
**
**      AUTHOR      = "Niel Kempson",
**      ADDRESS     = "25 Whitethorn Drive
**                     Cheltenham
**                     GL52 5LL
**                     England",
**      TELEPHONE   = "+44-242 579105",
**      EMAIL       = "kempson@tex.ac.uk (Internet)",
**
**      SUPPORTED   = "yes",
**      ARCHIVED    = "tex.ac.uk, ftp.tex.ac.uk",
**      KEYWORDS    = "VVcode",
**
**      CODETABLE   = "ISO/ASCII",
**      CHECKSUM    = "51492 1481 5732 57976",
**
**      DOCSTRING   = { This file is part of VVcode.
**                  }
**  }
**
**  MODULE CONTENTS
**
**      apply_defaults      -   Apply default file name and extension to a
**                              full file specification if either of these 
**                              components is missing.
**      confirm_yesno       -   Display a message to the user and wait for a
**                              yes/no answer.
**      examine_file        -   Examine a file and determine its key features, 
**                              including mode, format, record length and 
**                              timestamp.
**      explain_error       -   Explain the reason for an error.
**      f_open_in           -   Open a file for input using the appropriate 
**                              mode, format etc for the file.
**      f_open_out          -   Open a file for output using the appropriate
**                              mode, format etc for the file.
**      file_exists         -   Determine whether a file exists.
**      force_file_ext      -   Force the file extension of a full file
**                              specification to a specified value.
**      is_a_file           -   Determine whether a file stream is connected
**                              to a real file rather than a character
**                              device, pipe etc.
**      legal_filespec      -   Takes an arbitrary string which may a file 
**                              specification from another operating system
**                              and manipulates it to produce a legal file 
**                              specification for the current operating
**                              system.
**      make_filespec       -   Construct a full file specification from
**                              component parts.
**      prompt_for_string   -   Present a prompt string and accept a string 
**                              from the user.
**      read_bytes          -   Read bytes from a currently open file.
**      read_line           -   Read a line from a currently open (text) file.
**      read_record         -   Read a record from a currently open file.
**      scan_cmd_line       -   [tbs]
**      set_ftime           -   Set the timestamp of a file to a specified 
**                              value.
**      set_pipe_mode       -   Set the mode of a file stream connected to a
**                              pipe, character device, redirected
**                              stdin/stdout/stderr, in fact any non-file.
**      split_file_spec     -   Split a full file specification into its
**                              component parts.
**      tz_offset           -   Determine the offset of local time from 
**                              Greenwich Mean Time (Coordinated Universal
**                              Time) at a specified date and time.  
**      user_message        -   Present a message to the user.
**      vv_exit             -   Exit the program, returning the appropriate
**                              status to the operating system.
**      write_bytes         -   Write bytes to a currently open file.
**      write_record        -   Write a record to a currently open file.
**
**  COPYRIGHT
**
**      Written 1991-1992 by Niel Kempson <kempson@tex.ac.uk> who claims
**      no copyright over this program - you are free to use it as you wish.
**
**  CHANGE LOG
**
******************************************************************************
******************************************************************************
*/
static char rcsid[] = "$Id$";

/*-
**----------------------------------------------------------------------------
** Standard include files
**----------------------------------------------------------------------------
*/
#include <stdio.h>

/*-
**----------------------------------------------------------------------------
** Include files
**----------------------------------------------------------------------------
*/
#include "checkos.h"
#include "machine.h"
#include "local.h"
#include "globals.h"
#include "specific.h"
#include "vvutils.h"

/*-
**----------------------------------------------------------------------------
** Implementation (compiler) specific include files.  The default 
**----------------------------------------------------------------------------
*/
# if (UNIX_BSD)
#  include <sys/timeb.h>

#  ifndef HAS_UTIME_H
#   undef HAS_SYSUTIME_H
#   define HAS_SYSUTIME_H           1
#  endif                        /* HAS_UTIME_H */
# endif                         /* (UNIX_BSD) */

# if (UNIX_SYSV)
#  ifndef HAS_SYSUTIME_H
#   undef HAS_UTIME_H
#   define HAS_UTIME_H              1
#  endif
# endif                         /* UNIX_SYSV */

#ifdef HAS_UTIME_H
# include <utime.h>
#endif                          /* HAS_UTIME_H */

#ifdef HAS_SYSUTIME_H
# include <sys/utime.h>
#endif                          /* HAS_SYSUTIME_H */



/*-
**============================================================================
**
** FUNCTION
**
**      apply_defaults
**
** DESCRIPTION
**
**      Apply default file name and extension to a full file specification if
**      either of these components is missing.
**
** INPUT PARAMETERS
** 
**      full_spec       -   a string buffer of size (MAX_PATH + 1) containing
**                          the full file specification.
**      default_name    -   the default name component to be used if the full
**                          file specification does not contain a name
**                          component.  If there is no default name component,
**                          set this parameter to the null character pointer 
**                          defined by NULL.
**      default_ext     -   the default extension component to be used if the 
**                          full file specification does not contain an
**                          extension component.  If there is no default
**                          extension component, set this parameter to the 
**                          null character pointer defined by NULL.
**                          
** OUTPUT PARAMETERS
**
**      full_spec       -   a string buffer of size (MAX_PATH + 1), which on
**                          return will contain the full file specification
**                          with default name and extension components applied
**                          if appropriate.
**
** RETURN VALUE
**
**      None
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    void                    apply_defaults (CONST char *default_name,
                                            CONST char *default_ext,
                                            char *full_spec)
#else                           /* NOT (ANSI_SYNTAX) */
    void                    apply_defaults (default_name, default_ext, 
                                            full_spec)
        CONST char             *default_name;
        CONST char             *default_ext;
        char                   *full_spec;
#endif                          /* (ANSI_SYNTAX) */
{
    char                    name[MAX_NAME + 1];
    char                    extension[MAX_EXT + 1];
    Unsigned16              path_parts;
    char                    preamble[MAX_PREAMBLE + 1];

    path_parts = split_file_spec (full_spec, preamble, name, extension, NULL);


    if (((path_parts & FS_NAME) == 0) && (default_name != NULL))
    {
        strncpy (name, default_name, MAX_NAME);
    }

    if (((path_parts & FS_EXTENSION) == 0) && (default_ext != NULL))
    {
        strncpy (extension, default_ext, MAX_EXT);
    }

    make_file_spec (preamble, name, extension, NULL, full_spec);
}                               /* apply_defaults */



/*-
**============================================================================
**
** FUNCTION
**
**      confirm_yesno
**
** DESCRIPTION
**
**      Display a message to the user and wait for a yes/no answer.
**
** INPUT PARAMETERS
**
**      confirm_str     -   a string containing the message to be displayed.
**
** OUTPUT PARAMETERS
** 
**      None
**
** RETURN VALUE
**
**      TRUE if 'y' or 'Y' was entered, FALSE if 'n' or 'N' was entered.
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    Boolean                 confirm_yesno (CONST char *confirm_str)
#else                           /* NOT (ANSI_SYNTAX) */
    Boolean                 confirm_yesno (confirm_str)
        CONST char             *confirm_str;
#endif                          /* (ANSI_SYNTAX) */
{
    int                     answer;

    FPRINTF (stderr, "\n%s [Y/N]: ", confirm_str);
    fflush (stderr);

    do
    {
        answer = getchar ();

        if (answer == EOF)
        {
            ERRORMSG ("EOF detected on standard input");
            answer = 'N';
        }
        else
        {
            answer = TOUPPER (answer);
        }
    } while (!((answer == 'N') || (answer == 'Y')));

    return ((answer == 'Y') ? (Boolean) TRUE : (Boolean) FALSE);
}                               /* confirm_yesno */



/*-
**============================================================================
**
** FUNCTION
**
**      examine_file
**
** DESCRIPTION
**
**      Examine a file and determine its key features, including mode, format,
**      record length and timestamp.  The file information is passed via a 
**      File_Info structure.  The following fields of the structure are used:
**
** INPUT PARAMETERS
**
**      ex_file         -   the File_Info structure containing the feature
**                          information of the file to be examined.  The
**                          following structure fields are used for input:
**
**        .file_spec    -   the file specification of the file to be examined
**
** OUTPUT PARAMETERS
** 
**      ex_file         -   the File_Info structure whose elements will be
**                          updated to reflect the file features.  The
**                          following structure fields are used for output:
**
**        .mode         -   the mode of the file to be examined (binary/text).
**                          If the mode cannot be determined it shall be set
**                          to INV_MODE.
**        .format       -   the format of the file to be examined (fixed,
**                          stream or variable).  If the format cannot be 
**                          determined or is not supported, it shall be set
**                          to INV_FMT.
**        .max_rec_len  -   the maximum allowable record length.  If the
**                          maximum allowable record length cannot be
**                          determined or is not supported, it shall be set
**                          to INV_RECORD_LEN.
**        .lng_rec_len  -   the length of the longest record in the file.  If 
**                          the longest record length cannot be determined or
**                          is not supported, it shall be set to 
**                          INV_RECORD_LEN.
**        .mod_time    -    the modification time of the file in Unix format.
**                          If the time cannot be determined or is not
**                          supported, it shall be set to INV_TIMESTAMP.
**
** RETURN VALUE
**
**      None
**
**============================================================================
*/
#define EX_SAMPLE_SIZE         512

#if (ANSI_SYNTAX)
    void                    examine_file (File_Info *ex_file)
#else                           /* NOT (ANSI_SYNTAX) */
    void                    examine_file (ex_file)
        File_Info              *ex_file;
#endif                          /* (ANSI_SYNTAX) */
{
    char                    tmp_buffer[EX_SAMPLE_SIZE];
    int                     bytes_read;
    int                     ctrl_count;
    int                     eol_count;
    int                     i;
    struct stat             stat_buffer;

    /*-
    **------------------------------------------------------------------------
    ** Initialize the File_Info structure before starting work.  If the input
    ** file is not a real file (i.e. a device or pipe), these are the values
    ** that will be returned.
    **------------------------------------------------------------------------
    */
    ex_file->mode = INV_MODE;
    ex_file->format = INV_FORMAT;
    ex_file->max_rec_len = INV_RECORD_LEN;
    ex_file->lng_rec_len = INV_RECORD_LEN;
    ex_file->mod_time = INV_TIMESTAMP;

    /*-
    **------------------------------------------------------------------------
    **
    **------------------------------------------------------------------------
    */
    ex_file->file_ptr = fopen (ex_file->file_spec, "r");

    if (ex_file->file_ptr == (FILE *) NULL)
    {
        ERRORMSG_1 ("couldn't examine (open) file `%s'", ex_file->file_spec);
        explain_error ();
        return;
    }

    /*-
    **------------------------------------------------------------------------
    ** Test to see if the file is 'real'.
    **------------------------------------------------------------------------
    */
    if (is_a_file (ex_file) == FALSE)
    {
        DEBUG_1 ("examine_file: `%s' is not a regular file",
                 ex_file->file_spec);
        f_close (ex_file);
        ex_file->pipe_flag = TRUE;
        ex_file->file_ptr = (FILE *) NULL;
        return;
    }

    /*-
    **------------------------------------------------------------------------
    **
    **------------------------------------------------------------------------
    */
    bytes_read = fread (tmp_buffer, sizeof (char), EX_SAMPLE_SIZE,
                        ex_file->file_ptr);
    eol_count = 0;
    ctrl_count = 0;
    
    for (i = 0; i < (bytes_read - 1); i++)
    {
        if (tmp_buffer[i + 1] == '\n')
        {
            ++eol_count;
        }
        if (tmp_buffer[i + 1] < 0x08)
        {
            ++ctrl_count;
        }
    }
    f_close (ex_file);
    ex_file->file_ptr = (FILE *) NULL;

    /*-
    **------------------------------------------------------------------------
    **
    **------------------------------------------------------------------------
    */
    if ((bytes_read > (80 * eol_count)) || (bytes_read < (80 * ctrl_count)))
    {
        ex_file->mode = MODE_BINARY;
        ex_file->format = DEF_BINARY_FMT;
    }
    else
    {
        ex_file->mode = MODE_TEXT;
        ex_file->format = DEF_TEXT_FMT;
    }

    /*-
    **------------------------------------------------------------------------
    **
    **------------------------------------------------------------------------
    */
    i = stat (ex_file->file_spec, &stat_buffer);

    if (i != FALSE)
    {
        ERRORMSG_1 ("couldn't examine (stat) file `%s'", ex_file->file_spec);
        explain_error ();
        return;
    }
    ex_file->mod_time = stat_buffer.st_mtime;

}                               /* examine_file */



/*-
**============================================================================
**
** FUNCTION
**
**      explain_error
**
** DESCRIPTION
**
**      Explain the reason for a detected error.
**
** INPUT PARAMETERS
**
**      None
**
** OUTPUT PARAMETERS
** 
**      None
**
** RETURN VALUE
**
**      None
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    void                    explain_error (void)
#else                           /* NOT (ANSI_SYNTAX) */
    void                    explain_error ()
#endif                          /* (ANSI_SYNTAX) */
{
    INFOMSG_1 ("Error number:          %ld", (long) errno);
}                               /* explain_error */



/*-
**============================================================================
**
** FUNCTION
**
**      f_open_in
**
** DESCRIPTION
**
**      Open a file for input using the appropriate mode, format etc for the
**      file.
**
** INPUT PARAMETERS
**
**      ip_file       -     the File_Info structure whose elements will be
**                          used to open the file.  The following structure 
**                          fields are used for input:
**
**        .mode         -   the mode of the file (binary/text).
**        .format       -   the format of the file (fixed, stream or variable).
**        .max_rec_len  -   the maximum allowable record length.  If this
**                          field is set to INV_RECORD_LEN, a sensible default
**                          shall be chosen if needed and possible.
**        .lng_rec_len  -   the length of the longest record in the file.  If
**                          this field is set to INV_RECORD_LEN, a sensible
**                          default shall be chosen if needed and possible.
**
** OUTPUT PARAMETERS
** 
**      None
**
** RETURN VALUE
**
**      If the file was opened successfully, a (FILE *) pointer is returned,
**      otherwise (FILE *) NULL in the event of failure.
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    FILE                   *f_open_in (File_Info *ip_file)
#else                           /* NOT (ANSI_SYNTAX) */
    FILE                   *f_open_in (ip_file)
        File_Info              *ip_file;
#endif                          /* (ANSI_SYNTAX) */
{
    FILE                   *file_ptr;

    /*-
    **------------------------------------------------------------------------
    **
    **------------------------------------------------------------------------
    */
    switch (ip_file->mode)
    {
        case MODE_BINARY:
        case MODE_TEXT:
            file_ptr = fopen (ip_file->file_spec, "rb");
            break;
        default:
            ERRORMSG_1 ("invalid file mode specified (%d)", ip_file->mode);
            return ((FILE *) NULL);
    }

    if (file_ptr == (FILE *) NULL)
    {
        ERRORMSG_1 ("error opening file `%s' for input", ip_file->file_spec);
        explain_error ();
        return ((FILE *) NULL);
    }
    return (file_ptr);
}                               /* f_open_in */



/*-
**============================================================================
**
** FUNCTION
**
**      f_open_out
**
** DESCRIPTION
**
**      Open a file for output using the appropriate mode, format etc for the
**      file.
**
** INPUT PARAMETERS
**
**      op_file       -     the File_Info structure whose elements will be
**                          used to open the file.  The following structure 
**                          fields are used for input:
**
**        .mode         -   the mode of the file (binary/text).
**        .format       -   the format of the file (fixed, stream or variable).
**        .max_rec_len  -   the maximum allowable record length.  If this
**                          field is set to INV_RECORD_LEN, a sensible default
**                          shall be chosen if needed and possible.
**        .lng_rec_len  -   the length of the longest record in the file.  If
**                          this field is set to INV_RECORD_LEN, a sensible
**                          default shall be chosen if needed and possible.
**
**      overwrite_files -   flag to control the overwriting of existing files.
**                          If TRUE, existing files will be overwritten,
**                          otherwise the user will be asked to confirm that
**                          the files should be overwritten.
** 
** OUTPUT PARAMETERS
** 
**      None
**
** RETURN VALUE
**
**      If the file was opened successfully, a (FILE *) pointer is returned,
**      otherwise (FILE *) NULL in the event of failure.
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    FILE                   *f_open_out (CONST Int16 overwrite_files,
                                        File_Info *op_file)
#else                           /* NOT (ANSI_SYNTAX) */
    FILE                   *f_open_out (overwrite_files, op_file)
        CONST Int16             overwrite_files;
        File_Info              *op_file;
#endif                          /* (ANSI_SYNTAX) */
{
    FILE                   *file_ptr;

    /*-
    **------------------------------------------------------------------------
    **
    **------------------------------------------------------------------------
    */
    if ((overwrite_files != OVERWRITE_YES)
        && (file_exists (op_file->file_spec)))
    {
        if (overwrite_files == OVERWRITE_ASK)
        {
            SPRINTF (G_tmp_msg, "File `%s' exists.  Overwrite ?",
                     op_file->file_spec);

            if (!confirm_yesno (G_tmp_msg))
            {
                INFOMSG ("operation cancelled");
                return ((FILE *) NULL);
            }
        }
        else
        {
            ERRORMSG_1 ("file `%s' exists", op_file->file_spec);
            return ((FILE *) NULL);
        }
    }

    /*-
    **------------------------------------------------------------------------
    **
    **------------------------------------------------------------------------
    */
    switch (op_file->mode)
    {
        case MODE_BINARY:
        case MODE_TEXT:
            file_ptr = fopen (op_file->file_spec, "wb");
            break;
        default:
            ERRORMSG_1 ("invalid file mode specified (%d)", op_file->mode);
            return ((FILE *) NULL);
    }

    if (file_ptr == (FILE *) NULL)
    {
        ERRORMSG_1 ("error opening file `%s' for output", op_file->file_spec);
        explain_error ();
        return ((FILE *) NULL);
    }
    return (file_ptr);
}                               /* f_open_out */



/*-
**============================================================================
**
** FUNCTION
**
**      file_exists
**
** DESCRIPTION
**
**      Determine whether a file exists.  Access privileges (e.g. R, RW) are
**      not checked.  
**
** INPUT PARAMETERS
** 
**      file_spec       -   the file specification of the file to be checked.
**
** OUTPUT PARAMETERS
**
**      None
**
** RETURN VALUE
**
**      TRUE if the file exists, FALSE if not.
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    Boolean                 file_exists (CONST char *file_spec)
#else                           /* NOT (ANSI_SYNTAX) */
    Boolean                 file_exists (file_spec)
        CONST char             *file_spec;
#endif                          /* (ANSI_SYNTAX) */
{
    return ((Boolean) (access ((char *) file_spec, 0) == 0));
}                               /* file_exists */



/*-
**============================================================================
**
** FUNCTION
**
**      force_file_ext
**
** DESCRIPTION
**
**      Force the file extension of a full file specification to a specified
**      value.
**
** INPUT PARAMETERS
** 
**      full_spec       -   a string buffer of size (MAX_PATH + 1) containing
**                          the full file specification.
**      forced_ext      -   the new extension component to be used in the 
**                          full file specification.  If there is no new
**                          extension component, set this parameter to the 
**                          null character pointer defined by NULL.
**                          
** OUTPUT PARAMETERS
**
**      full_spec       -   a string buffer of size (MAX_PATH + 1), which on
**                          return will contain the full file specification
**                          with the new extension component applied.
**
** RETURN VALUE
**
**      None
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    void                    force_file_ext (CONST char *forced_ext,
                                            char *full_spec)
#else                           /* NOT (ANSI_SYNTAX) */
    void                    force_file_ext (forced_ext, full_spec)
        CONST char             *forced_ext;
        char                   *full_spec;
#endif                          /* (ANSI_SYNTAX) */
{
    char                    name[MAX_NAME + 1];
    char                    preamble[MAX_PREAMBLE + 1];

    if (forced_ext == NULL)
    {
        return;
    }
    (void) split_file_spec (full_spec, preamble, name, NULL, NULL);
    make_file_spec (preamble, name, forced_ext, NULL, full_spec);
}                               /* force_file_ext */



/*-
**============================================================================
**
** FUNCTION
**
**      is_a_file
**
** DESCRIPTION
**
**      Determine whether an already open stream is connected to a real file
**      rather than a character mode device or pipe.
**
** INPUT PARAMETERS
** 
**      ex_file
**
** OUTPUT PARAMETERS
**
**      None
**
** RETURN VALUE
**
**      TRUE if the stream is connected to a real file, FALSE if not.
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    Boolean                 is_a_file (CONST File_Info *ex_file)
#else                           /* NOT (ANSI_SYNTAX) */
    Boolean                 is_a_file (ex_file)
        CONST File_Info        *ex_file;
#endif                          /* (ANSI_SYNTAX) */
{
    int                     file_handle;
    int                     status;

    file_handle = fileno (ex_file->file_ptr);
    status = isatty (file_handle);

    if (status == 0)
    {
        return (TRUE);
    }
    else
    {
        return (FALSE);
    }
}                               /* is_a_file */



/*-
**============================================================================
**
** FUNCTION
**
**      legal_filespec
**
** DESCRIPTION
**
**      Takes an arbitrary string which may a file specification from another
**      operating system and manipulates it to produce a legal file
**      specification for the current operating system.  The resultant file
**      specification must not include any preamble or postamble components
**      (i.e. name and/or extension only).
**
**      The string may be truncated, characters translated, or even untouched.
**
** INPUT PARAMETERS
** 
**      hdrf_spec       -   the arbitrary string which is usually a file
**                          specification from another operating system.
**
** OUTPUT PARAMETERS
**
**      None
**
** RETURN VALUE
**
**      A pointer to the legal file specification string.  The string is held
**      in a static buffer and will be overwritten by successive calls to this
**      function.
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    char                   *legal_filespec (CONST char *hdrf_spec)
#else                           /* NOT (ANSI_SYNTAX) */
    char                   *legal_filespec (hdrf_spec)
        CONST char             *hdrf_spec;
#endif                          /* (ANSI_SYNTAX) */
{
    char                    tmpf_name[MAX_NAME + 1];
    char                    tmpf_ext[MAX_EXT + 1];
    char                   *tmpf_spec;
    char                   *tmp_ptr;
    static char             opf_spec[MAX_PATH + 1];

    /*-
    **------------------------------------------------------------------------
    **
    **------------------------------------------------------------------------
    */
    tmpf_spec = STRDUP (hdrf_spec);
    tmp_ptr = STRCHR (tmpf_spec, '.');

    if (tmp_ptr != NULL)
    {
        strncpy (tmpf_ext, tmp_ptr, MAX_EXT);
        tmpf_ext[MAX_EXT] = '\0';
        *tmp_ptr = '\0';
    }
    else
    {
        strcpy (tmpf_ext, "");
    }
    strncpy (tmpf_name, tmpf_spec, MAX_NAME);
    tmpf_name[MAX_NAME] = '\0';

    /*-
    **------------------------------------------------------------------------
    **
    **------------------------------------------------------------------------
    */
    for (tmp_ptr = tmpf_name; *tmp_ptr != '\0'; tmp_ptr++)
    {
        if (STRCHR (LEGAL_FILESPEC_CHARS, *tmp_ptr) == NULL)
        {
            *tmp_ptr = REPLACEMENT_FILESPEC_CHAR;
        }
    }

    /*-
    **------------------------------------------------------------------------
    **
    **------------------------------------------------------------------------
    */
    for (tmp_ptr = tmpf_ext + 1; *tmp_ptr != '\0'; tmp_ptr++)
    {
        if (STRCHR (LEGAL_FILESPEC_CHARS, *tmp_ptr) == NULL)
        {
            *tmp_ptr = REPLACEMENT_FILESPEC_CHAR;
        }
    }

    /*-
    **------------------------------------------------------------------------
    **
    **------------------------------------------------------------------------
    */
    strcpy (opf_spec, tmpf_name);
    strcat (opf_spec, tmpf_ext);

    if (strcmp (hdrf_spec, opf_spec) != 0)
    {
        WARNMSG_2 ("suspicious header file spec `%s' changed to `%s'",
                   hdrf_spec, opf_spec);
    }
    return (opf_spec);
}                               /* legal_filespec */



/*-
**============================================================================
**
** FUNCTION
**
**      make_file_spec
**
** DESCRIPTION
**
**      Construct a full file specification from component parts.
**
** INPUT PARAMETERS
** 
**      preamble        -   the preamble component of the file specification.
**                          It shall not be necessary for the string to end 
**                          in one of the characters '\' or ':'.  If
**                          there is no name component, set this parameter to
**                          the null character pointer defined by NULL.
**      name            -   the name component of the file specification.  If
**                          there is no name component, set this parameter to
**                          the null character pointer defined by NULL.
**      extension       -   the extension component of the file
**                          specification.  It shall not be necessary for the
**                          string to begin with a '.'.  If there is no
**                          extension component, set this parameter to the
**                          null character pointer defined by NULL.
**      postamble       -   the postamble component of the file
**                          specification.  Unix doesn't support the
**                          postamble component of the VVcode file
**                          specification model so this parameter is ignored.
**                          
** OUTPUT PARAMETERS
**
**      full_spec       -   a string buffer of size (MAX_PATH + 1), which on
**                          return will contain the full file specification.
**
** RETURN VALUE
**
**      None
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    void                    make_file_spec (CONST char *preamble,
                                            CONST char *name,
                                            CONST char *extension,
                                            CONST char *postamble,
                                            char *full_spec)
#else                           /* NOT (ANSI_SYNTAX) */
    void                    make_file_spec (preamble, name, extension,
                                            postamble, full_spec)
        CONST char             *preamble;
        CONST char             *name;
        CONST char             *extension;
        CONST char             *postamble;
        char                   *full_spec;
#endif                          /* (ANSI_SYNTAX) */
{
    int                     len;
    char                   *tmp_ptr;

    /*-
    **------------------------------------------------------------------------
    ** Start with an empty string and append components of the file
    ** specification only if they exist.
    **------------------------------------------------------------------------
    */
    strcpy (full_spec, "");

    /*-
    **------------------------------------------------------------------------
    ** The preamble component corresponds to the Unix network node and/or path
    ** components in the form:
    **
    **      node:           node:dir1       node:/dir1      node:/dir1/
    **      node:/dir1/dir2 dir1            /dir1           /dir1/dir2
    **      
    **
    ** If the preamble component does not end in '/' or ':', '/' is appended
    ** only if the component is not empty.
    **------------------------------------------------------------------------
    */
    if ((preamble != NULL) && (*preamble != '\0'))
    {
        strncat (full_spec, preamble, MAX_PREAMBLE);
        len = strlen (preamble);
        tmp_ptr = STRCHR ("/:", preamble[len - 1]);

        if ((tmp_ptr == NULL) && (len < MAX_PREAMBLE))
        {
            strcat (full_spec, "/");
        }
    }

    /*-
    **------------------------------------------------------------------------
    ** Simply append the name if it is present.
    **------------------------------------------------------------------------
    */
    if (name != NULL)
    {
        strncat (full_spec, name, MAX_NAME);
    }

    /*-
    **------------------------------------------------------------------------
    ** If the extension component does not begin with '.', prepend '.' before
    ** appending the extension to the file specification.    
    **------------------------------------------------------------------------
    */
    if ((extension != NULL) && (*extension != '\0'))
    {
        if (*extension != '.')
        {
            strcat (full_spec, ".");
            strncat (full_spec, extension, MAX_EXT - 1);
        }
        else
        {
            strncat (full_spec, extension, MAX_EXT);
        }
    }
}                               /* make_file_spec */



/*-
**============================================================================
**
** FUNCTION
**
**      prompt_for_string
**
** DESCRIPTION
**
**      Display a message to the user and accept a string.
**
** INPUT PARAMETERS
**
**      prompt_str      -   a string containing the message to be displayed.
**      buf_size        -   the size of the buffer which will contain the
**                          returned string.
**
** OUTPUT PARAMETERS
** 
**      return_buffer   -   a buffer to contain the string entered by the
**                          user.
**
** RETURN VALUE
**
**      None
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    void                    prompt_for_string (CONST char *confirm_str,
                                               CONST Int16 buf_size,
                                               char *return_buffer)
#else                           /* NOT (ANSI_SYNTAX) */
    void                    prompt_for_string (confirm_str, buf_size,
                                               return_buffer)
        CONST char             *confirm_str;
        CONST Int16             buf_size;
        char                   *return_buffer;
#endif                          /* (ANSI_SYNTAX) */
{
    char                   *status;

    FPRINTF (stderr, "\n%s", confirm_str);
    fflush (stderr);
    status = fgets (return_buffer, (int) buf_size, stdin);

    if (status == NULL)
    {
        FATALMSG ("error reading string from stdin");
        explain_error ();
        vv_exit ();
    }

    status = STRCHR (return_buffer, '\n');
    if (status != NULL)
    {
        *status = '\0';
    }
}                               /* prompt_for_string */



/*-
**============================================================================
**
** FUNCTION
**
**      read_bytes
**
** DESCRIPTION
**
**      Read bytes from a currently open file.  This function is called to 
**      read stream and fixed length record format files.  If the operating 
**      system does not support stream or fixed length record format files,
**      this function will never be called, BUT IT MUST BE PRESENT to avoid
**      linker errors.
**
** INPUT PARAMETERS
** 
**      ip_file       -     the File_Info structure whose elements will be
**                          used to read the record.  The following structure
**                          fields are used for input:
**
**        .file_ptr     -   the FILE structure of the stream
**
**      max_bytes       -   the maximum number of bytes that may be read
**
** OUTPUT PARAMETERS
**
**      buffer          -   a buffer of size 'max_bytes' or longer which on
**                          return will contain the bytes read from the file
**
** RETURN VALUE
**
**       #              -   the number of bytes returned in the buffer
**                          (excluding the trailing '\0').
**      -1              -   on EOF
**      -2              -   if an error was detected
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    Int32                   read_bytes (CONST Int32 max_bytes,
                                        char *buffer,
                                        File_Info *ip_file)
#else                           /* NOT (ANSI_SYNTAX) */
    Int32                   read_bytes (max_bytes, buffer, ip_file)
        CONST Int32             max_bytes;
        char                   *buffer;
        File_Info              *ip_file;
#endif                          /* (ANSI_SYNTAX) */
{
    Int32                   idx;
    int                     temp;

    for (idx = 0L; idx < max_bytes; idx++)
    {
        temp = fgetc (ip_file->file_ptr);

        if (temp == EOF)
        {
            if (ferror (ip_file->file_ptr))
            {
                return (-2L);
            }

            if (idx == 0)
            {
                return (-1L);
            }
            break;
        }
        buffer[(SIZE_T) idx] = (char) temp;
    }
    return (idx);
}                               /* read_bytes */



/*-
**============================================================================
**
** FUNCTION
**
**      read_line
**
** DESCRIPTION
**
**      Read a line from a currently open file.  This function is called to 
**      read text files.
**
** INPUT PARAMETERS
** 
**      ip_file       -     the File_Info structure whose elements will be
**                          used to read the record.  The following structure
**                          fields are used for input:
**
**        .file_ptr     -   the FILE structure of the stream
**
**      max_bytes       -   the maximum number of bytes that may be read
**
** OUTPUT PARAMETERS
**
**      buffer          -   a buffer of size 'max_bytes' or longer which on
**                          return will contain the line read from the file
**
** RETURN VALUE
**
**       #              -   the number of bytes returned in the buffer
**                          (excluding the trailing '\0').
**      -1              -   on EOF
**      -2              -   if an error was detected
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    Int32                   read_line (CONST Int32 max_bytes,
                                       char *buffer,
                                       File_Info *ip_file)
#else                           /* NOT (ANSI_SYNTAX) */
    Int32                   read_line (max_bytes, buffer, ip_file)
        CONST Int32             max_bytes;
        char                   *buffer;
        File_Info              *ip_file;
#endif                          /* (ANSI_SYNTAX) */
{
    char                   *tmp_ptr;

    tmp_ptr = fgets (buffer, (int) max_bytes, ip_file->file_ptr);
    buffer[(SIZE_T) (max_bytes - 1)] = '\0';

    if (tmp_ptr == NULL)
    {
        if (ferror (ip_file->file_ptr))
        {
            return (-2L);
        }

        if (feof (ip_file->file_ptr))
        {
            return (-1L);
        }
    }
    return ((Int32) strlen (buffer));
}                               /* read_line */



/*-
**============================================================================
**
** FUNCTION
**
**      read_record
**
** DESCRIPTION
**
**      Read a record from a currently open file.  This function is only
**      called to read variable length format files.  If the operating system
**      does not support variable length format files, this function will
**      never be called, BUT IT MUST BE PRESENT to avoid linker errors.
**
** INPUT PARAMETERS
** 
**      ip_file       -     the File_Info structure whose elements will be
**                          used to read the record.  The following structure
**                          fields are used for input:
**
**        .file_ptr     -   the FILE structure of the stream
**
**      max_bytes       -   the maximum number of bytes that may be read
**
** OUTPUT PARAMETERS
**
**      buffer          -   a buffer of size 'max_bytes' or longer which on
**                          return will contain the record read from the file
**
** RETURN VALUE
**
**       #              -   the number of bytes returned in the buffer
**                          (including an appended '\n' characters but
**                          excluding the trailing '\0').
**      -1              -   on EOF
**      -2              -   if an error was detected
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    Int32                   read_record (CONST Int32 max_bytes,
                                         char *buffer,
                                         File_Info *ip_file)
#else                           /* NOT (ANSI_SYNTAX) */
    Int32                   read_record (max_bytes, buffer, ip_file)
        CONST Int32             max_bytes;
        char                   *buffer;
        File_Info              *ip_file;
#endif                          /* (ANSI_SYNTAX) */
{
    INTERNAL_ERROR ("read_record");
    return (-2L);
}                               /* read_record */



/*-
**============================================================================
**
** FUNCTION
**
**      scan_cmd_line
**
** DESCRIPTION
**
**      [tbs]
**
** INPUT PARAMETERS
** 
**      qargc           -   the number of command line arguments to be parsed
**                          (a copy of the main() argc parameter).
**      qargv           -   an array of strings containing the command line
**                          arguments to be parsed (a copy of the main() argv
**                          parameter).
**      q_intro_str     -   a string containing the valid single characters
**                          which may be used to introduce a command
**                          qualifier (e.g. "/-" for MS-DOS, "-" for Unix).
**      q_sep_str       -   a string containing the valid single characters
**                          which may be used to separate a command qualifier
**                          from its value (e.g. "=:-" for MS-DOS,
**                          " " for Unix).
**
** OUTPUT PARAMETERS
**
**      qual_array      -   the array of Qualifier_Struct structures which
**                          contain the status information for each of the
**                          valid command qualifiers
**      
** RETURN VALUE
**
**      None
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    void                    scan_cmd_line (CONST int qargc,
                                           CONST char *qargv[],
                                           CONST char *q_intro_str,
                                           CONST char *q_sep_str,
                                           Qualifier_Struct *qual_array,
                                           Qualifier_Struct *qual_ipfile,
                                           Qualifier_Struct *qual_opfile)
#else                           /* NOT (ANSI_SYNTAX) */
    void                    scan_cmd_line (qargc, qargv, q_intro_str,
                                           q_sep_str, qual_array, qual_ipfile,
                                           qual_opfile)
        CONST int               qargc;
        CONST char             *qargv[];
        CONST char             *q_intro_str;
        CONST char             *q_sep_str;
        Qualifier_Struct       *qual_array;
        Qualifier_Struct       *qual_ipfile;
        Qualifier_Struct       *qual_opfile;
#endif                          /* (ANSI_SYNTAX) */
{
    char                   *arg;
    int                     arg_no;
    int                     len;
    int                     qual_found;
    int                     q_no;
    char                   *qsep_ptr;
    char                   *qvalue_ptr;
    char                   *tmp_qstr_ptr;


    /*-
    **------------------------------------------------------------------------
    ** Loop through each of the user supplied command line arguments.
    **------------------------------------------------------------------------
    */
    tmp_qstr_ptr = NULL;

    for (arg_no = 1; arg_no < qargc; arg_no++)
    {
        arg = (char *) qargv[arg_no];

        /*-
        **--------------------------------------------------------------------
        **
        **--------------------------------------------------------------------
        */
        if (qual_array[CMDQ_DEBUG].present == TRUE)
        {
            G_debugging = TRUE;
        }
        DEBUG_1 ("scan_cmd_line: command line argument: `%s'", arg);

        /*-
        **--------------------------------------------------------------------
        ** If the first character of the argument is one of the allowable
        ** 'q_intro_str' characters then parse it as a qualifier, unless
        ** the argument is the 'STDIO_SYMBOL' string when it is interpreted as
        ** the standard input or output file.
        **--------------------------------------------------------------------
        */
        if (((STRCHR (q_intro_str, *arg) != NULL)
             && STRCMP (arg, STDIO_SYMBOL) != 0))
        {

            /*-
            **----------------------------------------------------------------
            ** Extract the component of the qualifier string after the
            ** 'q_intro_str' character before the 'q_sep_str' character
            ** if it is present.
            **----------------------------------------------------------------
            */
            if (tmp_qstr_ptr != NULL)
            {
                free (tmp_qstr_ptr);
            }
            tmp_qstr_ptr = STRDUP (arg + 1);
            qsep_ptr = STRSTR (tmp_qstr_ptr, q_sep_str);
            qvalue_ptr = NULL;

            if (qsep_ptr != NULL)
            {
                *qsep_ptr = '\0';
                qvalue_ptr = qsep_ptr + 1;
            }

            /*-
            **----------------------------------------------------------------
            ** Search the command qualifier structure array for this
            ** qualifier string.
            **----------------------------------------------------------------
            */
            len = strlen (tmp_qstr_ptr);
            qual_found = -1;

            for (q_no = 0; qual_array[q_no].q_string != NULL; q_no++)
            {
                if (STRNCMPI (qual_array[q_no].q_string, tmp_qstr_ptr, (Int16) len) == 0)
                {

                    /*-
                    **--------------------------------------------------------
                    **
                    **--------------------------------------------------------
                    */
                    if (qual_found < 0)
                    {
                        qual_found = q_no;
                        DEBUG_1 ("scan_cmd_line: matches qualifier:      `%s'",
                                 qual_array[q_no].q_string);
                    }
                    else
                    {
                        USAGE_1 ("ambiguous qualifier `%s'", qargv[arg_no]);
                    }
                }
            }

            /*-
            **----------------------------------------------------------------
            ** If the qualifier string is unknown, return that status.
            **----------------------------------------------------------------
            */
            if (qual_found < 0)
            {
                USAGE_1 ("unknown qualifier `%s'", qargv[arg_no]);
            }

            /*-
            **----------------------------------------------------------------
            **
            **----------------------------------------------------------------
            */
            if (qsep_ptr == NULL)
            {
                if (qual_array[qual_found].val_needed == VALUE_NEEDED)
                {
                    if ((arg_no + 1) < qargc)
                    {
                        qual_array[qual_found].present = TRUE;
                        qual_array[qual_found].value = STRDUP (qargv[++arg_no]);
                        DEBUG_1 ("scan_cmd_line: qualifier value:        `%s'",
                                 qual_array[qual_found].value);
                    }
                    else
                    {
                        USAGE_2 ("qualifier `%.1s%s' must take a value",
                                 QUAL_INTRO, qual_array[qual_found].q_string);
                    }
                }
                else
                {
                    qual_array[qual_found].present = TRUE;
                    qual_array[qual_found].value = NULL;
                }
            }
            else
            {
                /*-
                **------------------------------------------------------------
                **
                **------------------------------------------------------------
                */
                if (*qvalue_ptr == '\0')
                {
                    USAGE_1 ("value missing after qualifier `%s'",
                             qargv[arg_no]);
                }
                else if (qual_array[qual_found].val_needed == VALUE_NONE)
                {
                    USAGE_2 ("qualifier `%.1s%s' does not take a value",
                             QUAL_INTRO, qual_array[qual_found].q_string);
                }
                else
                {
                    qual_array[qual_found].value = STRDUP (qvalue_ptr);
                    qual_array[qual_found].present = TRUE;
                    DEBUG_1 ("scan_cmd_line: qualifier value:        `%s'",
                             qual_array[qual_found].value);
                }
            }
        }

        /*-
        **--------------------------------------------------------------------
        **
        **--------------------------------------------------------------------
        */
        else
        {
            if (qual_ipfile->present == FALSE)
            {
                qual_ipfile->present = TRUE;
                qual_ipfile->value = STRDUP (qargv[arg_no]);
            }
            else if (qual_opfile->present == FALSE)
            {
                qual_opfile->present = TRUE;
                qual_opfile->value = STRDUP (qargv[arg_no]);
            }
            else
            {
                USAGE ("too many file names specified");
            }
        }
    }
}                               /* scan_cmd_line */



/*-
**============================================================================
**
** FUNCTION
**
**      set_ftime
**
** DESCRIPTION
**
**      Set the timestamp of a file to a specified value.  Operating systems
**      support different types of time, such as access, creation, revision or
**      modification time.  It is suggested that the creation and modification
**      time be set to this value if possible.
**
**      If the operating system does not support the setting of file
**      timestamps, this should be an empty function, BUT IT MUST BE PRESENT.
**
** INPUT PARAMETERS
** 
**      target_file   -     the File_Info structure containing the feature
**                          information of the file whose timestamp is to be
**                          set.  The following structure fields are used
**                          for input:
**
**        .file_spec    -   the file specification of the file.
**        .mod_time     -   the modification time of the file in Unix format.
**
** OUTPUT PARAMETERS
**
**      None
**
** RETURN VALUE
**
**      None
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    void                    set_ftime (File_Info *target_file)
#else                           /* NOT (ANSI_SYNTAX) */
    void                    set_ftime (target_file)
        File_Info              *target_file;
#endif                          /* (ANSI_SYNTAX) */
{
    int                     status;

#if (UNIX_BSD)
    TIME_T                  timep[2];
#endif				/* (UNIX_BSD) */

#if (UNIX_SYSV)
    struct utimbuf          utb;
#endif				/* (UNIX_SYSV) */


    /*-
    **------------------------------------------------------------------------
    **
    **------------------------------------------------------------------------
    */
    DEBUG_2 ("set_ftime: setting file `%s' time to %.24s",
             target_file->file_spec, ctime (&(target_file->mod_time)));

    /*-
    **------------------------------------------------------------------------
    **
    **------------------------------------------------------------------------
    */
#if (UNIX_SYSV)
    utb.actime  = target_file->mod_time;
    utb.modtime = target_file->mod_time; 
    status = utime ((char *) target_file->file_spec, &utb);
#endif				/* (UNIX_SYSV) */

#if (UNIX_BSD)
    timep[0] = target_file->mod_time;  
    timep[1] = target_file->mod_time;  
    status = utime ((char *) target_file->file_spec, timep);
#endif				/* (UNIX_BSD) */


    /*-
    **------------------------------------------------------------------------
    **
    **------------------------------------------------------------------------
    */
    if (status != 0)
    {
        ERRORMSG_2 ("error setting file `%s' time to %.24s",
                    target_file->file_spec, ctime (&(target_file->mod_time)));
        explain_error ();
    }
    return;

}                               /* set_ftime */



/*-
**============================================================================
**
** FUNCTION
**
**      set_pipe_mode
**
** DESCRIPTION
**
**      Set the mode of an already open file stream which is a
**      character mode device or pipe rather than a real file.
**
**      If text/binary mode has no meaning or mode setting is not supported,
**      this function may be empty, BUT IT MUST BE PRESENT.
**
** INPUT PARAMETERS
** 
**      target_file   -     the File_Info structure containing the feature
**                          information of the file whose mode is to be
**                          changed.  The following structure fields are used
**                          for input:
**
**        .file_spec    -   the file specification of the open file.
**        .file_ptr     -   the (FILE *) pointer to the open file
**        .mode         -   the mode of the file to be examined (binary/text).
**
** OUTPUT PARAMETERS
**
**      None
**
** RETURN VALUE
**
**      None
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    void                    set_pipe_mode (File_Info *target_file)
#else                           /* NOT (ANSI_SYNTAX) */
    void                    set_pipe_mode (target_file)
        File_Info              *target_file;
#endif                          /* (ANSI_SYNTAX) */
{
    return;
}                               /* set_pipe_mode */



/*-
**============================================================================
**
** FUNCTION
**
**      split_file_spec
**
** DESCRIPTION
**
**      Split a full file specification into its component parts and return
**      a bit field indicating which components of the file specification
**      were present.
**
** INPUT PARAMETERS
**
**      full_spec       -   a string buffer of size (MAX_PATH + 1), which
**                          contains the full file specification.
**
** OUTPUT PARAMETERS
** 
**      preamble        -   a buffer of size (MAX_PREAMBLE + 1) to hold the
**                          preamble component of the file specification.
**                          The string shall end in one of the characters
**                          '/' or ':'.  If there is no preamble component, 
**                          this parameter shall be set to the null character
**                          pointer defined by NULL.
**      name            -   a buffer of size (MAX_NAME + 1) to hold the name
**                          component of the file specification.  If there is
**                          no name component, this parameter shall be set to
**                          the null character pointer defined by NULL.
**      extension       -   a buffer of size (MAX_EXT + 1) to hold the
**                          extension component of the file specification.
**                          The string shall begin with '.'  If there
**                          is no extension component, this parameter shall
**                          be set to the null character pointer defined by
**                          NULL.
**      postamble       -   a buffer of size (MAX_POSTAMBLE + 1) to hold the
**                          postamble component of the file specification.
**                          Unix doesn't support the postamble component of
**                          the VVcode file specification model so this 
**                          parameter is ignored.
**
** RETURN VALUE
**
**      An Int16 value indicating which components were found in the file
**      specification.  The value is composed of the following macros:
**
**          FS_NOTHING      -   no components were present (i.e. an empty file
**                              or null specification)
**          FS_PREAMBLE     -   a preamble component was present
**          FS_NAME         -   a name component was present
**          FS_EXTENSION    -   an extension component was present
**          FS_POSTAMBLE    -   a postamble component was present
**
**      If the full file specification contains more than one component, these
**      values are OR'd together.
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    Unsigned16              split_file_spec (CONST char *full_spec,
                                             char *preamble,
                                             char *name,
                                             char *extension,
                                             char *postamble)
#else                           /* NOT (ANSI_SYNTAX) */
    Unsigned16              split_file_spec (full_spec, preamble, name,
                                             extension, postamble)
        CONST char             *full_spec;
        char                   *preamble;
        char                   *name;
        char                   *extension;
        char                   *postamble;
#endif                          /* (ANSI_SYNTAX) */
{
    char                    tmp_full_spec[MAX_PATH + 1];
    char                   *start_ptr;
    char                   *tmp_ptr;
    int                     idx;
    Unsigned16              path_components;


    /*-
    **------------------------------------------------------------------------
    **
    **------------------------------------------------------------------------
    */
    if (preamble != NULL)
    {
        *preamble = '\0';
    }
    if (name != NULL)
    {
        *name = '\0';
    }
    if (extension != NULL)
    {
        *extension = '\0';
    }
    if (postamble != NULL)
    {
        *postamble = '\0';
    }
    path_components = FS_NOTHING;

    if ((full_spec == NULL) || (*full_spec == '\0'))
    {
        return (path_components);
    }

    strcpy (tmp_full_spec, full_spec);
    start_ptr = (char *) tmp_full_spec;

    /*-
    **------------------------------------------------------------------------
    **
    **------------------------------------------------------------------------
    */
    tmp_ptr = STRRSTR (start_ptr, ":/");

    if (tmp_ptr != NULL)
    {
        path_components |= FS_PREAMBLE;

        if (preamble != NULL)
        {
            for (idx = 0; ((idx < MAX_PREAMBLE) && (start_ptr <= tmp_ptr));
                 idx++)
            {
                *preamble++ = *start_ptr++;
            }
            *preamble = '\0';
        }
        start_ptr = ++tmp_ptr;
    }

    /*-
    **------------------------------------------------------------------------
    **
    **------------------------------------------------------------------------
    */
    tmp_ptr = STRCHR (start_ptr, '.');

    if (tmp_ptr != NULL)
    {
        path_components |= FS_EXTENSION;

        if (extension != NULL)
        {
            strncpy (extension, tmp_ptr, MAX_EXT);
        }
        *tmp_ptr = '\0';
    }

    /*-
    **------------------------------------------------------------------------
    **
    **------------------------------------------------------------------------
    */
    if (*start_ptr != '\0')
    {
        path_components |= FS_NAME;

        if (name != NULL)
        {
            strncpy (name, start_ptr, MAX_NAME);
        }
    }

    return (path_components);
}                               /* split_file_spec */



/*-
**============================================================================
**
** FUNCTION
**
**      tz_offset
**
** DESCRIPTION
**
**      Determine the offset of local time from Greenwich Mean Time
**      (Coordinated Universal Time) at a specified date and time.  The date
**      and time is specified because the offset may vary due to different
**      implementations of summer (daylight saving) time around the world.
**
**      If the operating system does not support the concept of time zones,
**      this function should return a zero offset.
**
** INPUT PARAMETERS
** 
**      the_time        -   the GMT time in Unix time_t format at which the
**                          offset is to be computed.
**
** OUTPUT PARAMETERS
**
**      None
**
** RETURN VALUE
**
**      The offset of local time from GMT in seconds.  Positive values denote
**      times WEST of (i.e. behind) GMT, and negative values are EAST (i.e.
**      ahead) of GMT.
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    TIME_T                  tz_offset (TIME_T the_time)
#else                           /* NOT (ANSI_SYNTAX) */
    TIME_T                  tz_offset (the_time)
        TIME_T                  the_time;
#endif                          /* (ANSI_SYNTAX) */
{
    struct tm              *tm;

    /*-
    **------------------------------------------------------------------------
    **
    **------------------------------------------------------------------------
    */
#if (UNIX_BSD)
    TIME_T                  timezone;
    struct timeb            timeb_struct;

    ftime (&timeb_struct);
    timezone = (TIME_T) timeb_struct.timezone * (TIME_T) 60;
#endif				/* (UNIX_BSD) */

    tm = localtime (&the_time);

    if (tm->tm_isdst > 0)
    {
        return (timezone - (TIME_T) 3600);
    }
    else
    {
        return (timezone);
    }
}                               /* tz_offset */



/*-
**============================================================================
**
** FUNCTION
**
**      user_message
**
** DESCRIPTION
**
**      [tbs]
**
** INPUT PARAMETERS
**
**      [tbs]
**
** OUTPUT PARAMETERS
**
**      [tbs]
**
** RETURN VALUE
**
**      [tbs]
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    void                    user_message (CONST Int16 status,
                                          CONST char *msg_str,
                                          File_Info *log_file)
#else                           /* NOT (ANSI_SYNTAX) */
    void                    user_message (status, msg_str, log_file)
        CONST Int16             status;
        CONST char             *msg_str;
        File_Info              *log_file;
#endif                          /* (ANSI_SYNTAX) */
{
    int                     to_stderr;
    int                     to_logfile;
    char                   *status_str;


    /*-
    **------------------------------------------------------------------------
    ** Check the value supplied for the message status and set the output
    ** destinations accordingly.
    **------------------------------------------------------------------------
    */
    to_logfile = TRUE;
    to_stderr = FALSE;

    switch (status)
    {
        case NORMAL_STATUS:
        case LOG_STATUS:
            status_str = "";
            break;
        case INFO_STATUS:
            status_str = " -- ";
            to_stderr = TRUE;
            break;
        case WARNING_STATUS:
            status_str = "\nWARNING: ";
            to_stderr = TRUE;
            break;
        case ERROR_STATUS:
            status_str = "\nERROR: ";
            to_stderr = TRUE;
            break;
        case FATAL_STATUS:
            status_str = "\nFATAL: ";
            to_stderr = TRUE;
            break;
        default:
            INTERNAL_ERROR ("user_message");
            vv_exit ();
    }

    /*-
    **------------------------------------------------------------------------
    ** If the message is NULL, just update the status and/or the message
    ** show status.
    **------------------------------------------------------------------------
    */
    if ((msg_str == NULL) || (*msg_str == '\0'))
    {
        if (status > G_status)
        {
            G_status = status;
        }
        return;
    }

    /*-
    **------------------------------------------------------------------------
    ** Make sure that we don't try to write to a NULL log file pointer.
    **------------------------------------------------------------------------
    */
    if (log_file->file_ptr == (FILE *) NULL)
    {
        to_logfile = FALSE;
    }

    /*-
    **------------------------------------------------------------------------
    ** If the message is to be sent to the log file and stderr, check whether
    ** the log file output is also to stderr.  If it is, disable the 'stderr'
    ** output.
    **------------------------------------------------------------------------
    */
    if (to_logfile && to_stderr)
    {
        if (log_file->file_ptr == stderr)
        {
            to_stderr = FALSE;
        }
    }

    /*-
    **------------------------------------------------------------------------
    ** If the message is to be sent to stderr, output it here.
    **------------------------------------------------------------------------
    */
    if (to_stderr)
    {
        FPRINTF (stderr, "%s%s\n", status_str, msg_str);
    }

    /*-
    **------------------------------------------------------------------------
    ** If the message is to be sent to the log file, output it here.  If
    ** debugging is enabled, report also the name of the calling function.
    **------------------------------------------------------------------------
    */
    if (to_logfile)
    {
        FPRINTF (log_file->file_ptr, "%s%s\n", status_str, msg_str);
    }

    /*-
    **------------------------------------------------------------------------
    **
    **------------------------------------------------------------------------
    */
    if (status > G_status)
    {
        G_status = status;
    }

}                               /* user_message */



/*-
**============================================================================
**
** FUNCTION
**
**      vv_exit
**
** DESCRIPTION
**
**      Exit the program, returning the appropriate status to the operating
**      system.
**
** INPUT PARAMETERS
** 
**      status          -   the VVcode status value which will be used to
**                          determine the exit status passed to the operating
**                          system
**
** OUTPUT PARAMETERS
**
**      None
**
** RETURN VALUE
**
**      None
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    void                    vv_exit (void)
#else                           /* NOT (ANSI_SYNTAX) */
    void                    vv_exit ()
#endif                          /* (ANSI_SYNTAX) */
{
    switch (G_status)
    {
        case LOG_STATUS:
        case NORMAL_STATUS:
            exit (0);
            break;
        case WARNING_STATUS:
            exit (1);
            break;
        case ERROR_STATUS:
            exit (2);
            break;
        case FATAL_STATUS:
        default:
            exit (3);
            break;
    }
}                               /* vv_exit */



/*-
**============================================================================
**
** FUNCTION
**
**      write_bytes
**
** DESCRIPTION
**
**      Write bytes to a currently open file.
**
** INPUT PARAMETERS
** 
**      op_file       -     the File_Info structure whose elements will be
**                          used to write the record.  The following structure
**                          fields are used for input:
**
**        .file_ptr     -   the FILE structure of the stream
**        .max_rec_len  -   the maximum allowable record length.
**
**      buffer          -   a buffer of size 'rec_len' or longer which
**                          contains the record to be written to the file
**      rec_len         -   the number of bytes in the record
**
** OUTPUT PARAMETERS
**
**      None
**
** RETURN VALUE
**
**      None
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    void                    write_bytes (CONST Int32 max_bytes,
                                         CONST char *buffer,
                                         File_Info *op_file)
#else                           /* NOT (ANSI_SYNTAX) */
    void                    write_bytes (max_bytes, buffer, op_file)
        CONST Int32             max_bytes;
        CONST char             *buffer;
        File_Info              *op_file;
#endif                          /* (ANSI_SYNTAX) */
{
    Int32                   idx;

    for (idx = 0L; idx < max_bytes; idx++)
    {
        fputc (buffer[(SIZE_T) idx], op_file->file_ptr);
    }
}                               /* write_bytes */



/*-
**============================================================================
**
** FUNCTION
**
**      write_record
**
** DESCRIPTION
**
**      Write a record to a currently open file.  This function is only
**      called to write variable length format files.  If the operating
**      system does not support variable length format files, this function
**      will never be called, BUT IT MUST BE PRESENT to avoid linker errors.
**
** INPUT PARAMETERS
** 
**      op_file       -     the File_Info structure whose elements will be
**                          used to write the record.  The following structure
**                          fields are used for input:
**
**        .file_ptr     -   the FILE structure of the stream
**        .max_rec_len  -   the maximum allowable record length.
**
**      buffer          -   a buffer of size 'rec_len' or longer which
**                          contains the record to be written to the file
**      rec_len         -   the number of bytes in the record
**
** OUTPUT PARAMETERS
**
**      None
**
** RETURN VALUE
**
**      None
**
**============================================================================
*/
#if (ANSI_SYNTAX)
    void                    write_record (CONST Int32 max_bytes,
                                          CONST char *buffer,
                                          File_Info *op_file)
#else                           /* NOT (ANSI_SYNTAX) */
    void                    write_record (max_bytes, buffer, op_file)
        CONST Int32             max_bytes;
        CONST char             *buffer;
        File_Info              *op_file;
#endif                          /* (ANSI_SYNTAX) */
{
    INTERNAL_ERROR ("write_record");
}                               /* write_record */
